module net.BurtonRadons.parse.error;

private import net.BurtonRadons.parse.lexer;

/** The base error class of all dparser errors. */
class MarkedError : Error
{
    Marker mark; /**< Marked region relevant to the error. */

    /** Assign the marker and the std.string. */
    this (Marker mark, char [] msg)
    {
        super (mark.message () ~ msg);
        this.mark = mark;
    }
}

/** A character was not understood by the lexer. */
class InvalidCharacterError : MarkedError
{
    dchar ch; /**< Character that failed. */

    this (Marker mark, dchar ch)
    {
        char [] m;

        if (ch >= 32 && ch < 128)
            m = fmt ("Invalid character '%c'.", ch);
        else if (ch > 255)
            m = fmt ("Invalid character 0x%04X.", (uint) ch);
        else if (ch != "'")
            m = fmt ("Invalid character 0x%02X.", (char) ch);
        else
            m = fmt ("Invalid character \"'\".", ch);
        super (mark, m);
    }
}

/** Abstract base of the UnterminatedMultilineCommentError and UnterminatedNestedCommentError. */
class UnterminatedCommentError : MarkedError
{
    this (Marker mark, char [] msg)
    {
        super (mark, msg);
    }
}

/** Unterminated multiline comment. */
class UnterminatedMultilineCommentError : UnterminatedCommentError
{
    this (Marker mark)
    {
        super (mark, "Unterminated multiline comment; required \"*/\" to end the comment.");
    }
}

/** Unterminated nested comment. */
class UnterminatedNestedCommentError : UnterminatedCommentError
{
    int depth; /**< Number of +/ needed to escape the comment. */

    this (Marker mark, int depth)
    {
        char [] m;

        this.depth = depth;
        if (depth == 1)
            m = fmt ("Unterminated nested comment; required \"+/\" to end the comment.");
        else
            m = fmt ("Unterminated nested comment; required %d \"+/\" tokens to end the comment.", depth);
        super (mark, m);
    }
}

/** Unterminated double-quoted std.string. */
class UnterminatedDoublequoteStringError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("Unterminated double-quoted string; required \"\"\" to end the std.string."));
    }
}

/** Unterminated single-quoted std.string. */
class UnterminatedSinglequoteStringError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("Unterminated single-quoted string; required \"\'\" to end the std.string."));
    }
}

/** End-of-file after escape character. */
class EOFAfterEscapeError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("Escape character (\"\\\") followed by end of file; expected escape value."));
    }
}

/** Ill-formed escape code. */
class IllformedEscapeError : MarkedError
{
    this (Marker mark, char [] msg)
    {
        super (mark, msg);
    }
}

/** An invalid token appeared at a certain scope. */
class UnhandledTokenError : MarkedError
{
    this (Marker mark, char [] msg)
    {
        super (mark, msg);
    }
}

/** Expected one token, got the quoted token. */
class ExpectTokenError : MarkedError
{
    char [] expected; /**< The token that was expected. */
    char [] received; /**< The token that was received. */

    this (Marker mark, char [] expected, char [] received)
    {
        this.expected = expected;
        this.received = received;
        super (mark, fmt ("Expected %.*s but got %.*s.", expected, received));
    }
}

/** The calling convention in "extern (XXX)" was not known. */
class UnknownCallingConventionError : MarkedError
{
    char [] name; /**< The calling convention tried. */

    this (Marker mark, char [] name)
    {
        this.name = name;
        super (mark, fmt ("Unknown calling convention %.*s.", name));
    }
}

/** Didn't expect an identifier. */
class UnexpectedIdentifierError : MarkedError
{
    char [] id; /**< The identifier we didn't expect. */

    this (Marker mark, char [] id)
    {
        this.id = id;
        super (mark, fmt ("Did not expect identifier %.*s.", id));
    }
}

/** Expected an identifier but didn't get one. */
class ExpectedIdentifierError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("Expected identifier."));
    }
}

/** Array brackets were given after the name. */
class PostNameBracketError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("C-style array declarations are not supported; move the bracket portion before the name."));
    }
}

/** The semicolon was used as an empty statement. */
class SemicolonStatementError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("Use \"{ }\" to indicate an empty statement, not \";\"."));
    }
}

/** Cannot implicitly cast from one type to another. */
class CannotImplicitCastError : MarkedError
{
    this (Marker mark, char [] from, char [] to)
    {
        super (mark, fmt ("Cannot implicitly cast from %.*s to %.*s.", from, to));
    }
}

/** Could not find a named identifier. */
class UndefinedIdentifierError : MarkedError
{
    this (Marker mark, char [] name)
    {
        super (mark, fmt ("Undefined identifier %.*s.", name));
    }
}

/** "this" was used outside of any function declaration. */
class ThisOutsideFunctionError : MarkedError
{
    this (Marker mark)
    {
        super (mark, fmt ("\"this\" is only allowed inside method declarations."));
    }
}

/** "this" was used in a non-member function. */
class NoThisInFunctionError : MarkedError
{
    this (Marker mark, char [] name)
    {
        super (mark, fmt ("Function \"%.*s\" is not a method, so \"this\" is undefined.", name));
    }
}

/** An "abstract" function has been given a body. */
class AbstractWithBodyError : MarkedError
{
    this (Marker mark, char [] name)
    {
        super (mark, fmt ("\"abstract\" method \"%.*s\" cannot have a body.", name));
    }
}